#!/bin/bash

# Получение аргументов
while getopts d: flag
do
    case "${flag}" in
        d) DOMAIN=${OPTARG};;
    esac
done

# Проверка, что домен передан
if [ -z "$DOMAIN" ]; then
    echo "Укажите домен с помощью -d."
    exit 1
fi

# Функция для проверки установки пакета
function check_dependency {
    if ! command -v $1 &> /dev/null; then
        echo "Устанавливается $1..."
        dnf install -y $1
    else
        echo "$1 уже установлен."
    fi
}

# Проверка зависимостей
check_dependency "curl"
check_dependency "unzip"
check_dependency "nginx"
check_dependency "php"
check_dependency "php-fpm"
check_dependency "redis"

# Установка EPEL и Remi репозиториев
dnf install -y epel-release
dnf install -y https://rpms.remirepo.net/enterprise/remi-release-9.rpm
dnf module reset php
dnf module enable -y php:remi-8.3

# Установка PHP и его расширений
dnf install -y php php-cli php-fpm php-json php-pdo php-mbstring php-tokenizer php-xml php-curl php-zip php-bcmath php-gd php-mysqlnd php-pecl-redis

# Установка и настройка MariaDB
dnf install -y mariadb-server mariadb
systemctl start mariadb
systemctl enable mariadb

# Настройка MariaDB для высоконагруженных систем
cat > /etc/my.cnf.d/server.cnf << EOL
[mysqld]
max_connections = 1000
innodb_buffer_pool_size = 1G
innodb_log_file_size = 256M
innodb_flush_log_at_trx_commit = 2
max_allowed_packet = 64M
query_cache_size = 0
query_cache_type = 0
tmp_table_size = 64M
EOL
systemctl restart mariadb

# Создание базы данных
DB_ROOT_PASSWORD=$(openssl rand -base64 12)
DB_NAME="laravel"
DB_USER="laravel_user"
DB_PASSWORD=$(openssl rand -base64 12)

mysql -u root <<MYSQL_SCRIPT
CREATE DATABASE $DB_NAME;
CREATE USER '$DB_USER'@'localhost' IDENTIFIED BY '$DB_PASSWORD';
GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USER'@'localhost';
FLUSH PRIVILEGES;
MYSQL_SCRIPT

# Скачивание и распаковка Laravel проекта
ZIP_URL="https://example.com/laravel_project.zip"
curl -o laravel_project.zip $ZIP_URL
mkdir -p /var/www/$DOMAIN
unzip laravel_project.zip -d /var/www/$DOMAIN
rm laravel_project.zip

# Настройка прав
chown -R nginx:nginx /var/www/$DOMAIN
chmod -R 775 /var/www/$DOMAIN/storage
chmod -R 775 /var/www/$DOMAIN/bootstrap/cache

# Настройка Nginx
cat > /etc/nginx/conf.d/$DOMAIN.conf << EOL
server {
    listen 80;
    server_name $DOMAIN;
    root /var/www/$DOMAIN/public;

    index index.php index.html index.htm;

    location / {
        try_files \$uri \$uri/ /index.php?\$query_string;
    }

    location ~ \.php$ {
        include fastcgi_params;
        fastcgi_pass 127.0.0.1:9000;
        fastcgi_param SCRIPT_FILENAME \$document_root\$fastcgi_script_name;
        fastcgi_param PATH_INFO \$fastcgi_path_info;
    }

    location ~ /\.ht {
        deny all;
    }
}
EOL
systemctl restart nginx

# Установка и настройка Redis
dnf install -y redis
systemctl start redis
systemctl enable redis

# Настройка .env
APP_KEY=$(php /var/www/$DOMAIN/artisan key:generate --show)

cat > /var/www/$DOMAIN/.env << EOL
APP_NAME=Laravel
APP_ENV=production
APP_KEY=$APP_KEY
APP_DEBUG=false
APP_URL=http://$DOMAIN

DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=$DB_NAME
DB_USERNAME=$DB_USER
DB_PASSWORD=$DB_PASSWORD

CACHE_DRIVER=redis
QUEUE_CONNECTION=redis
REDIS_HOST=127.0.0.1
REDIS_PASSWORD=null
REDIS_PORT=6379
EOL

# Миграции и сиды
cd /var/www/$DOMAIN
php artisan migrate --seed

echo "Установка завершена."
echo "Домен: http://$DOMAIN"
echo "DB_USER: $DB_USER"
echo "DB_PASSWORD: $DB_PASSWORD"
echo "ROOT DB:" $DB_ROOT_PASSWORD
