#!/usr/bin/env python3
import argparse
import paramiko
import mysql.connector
from mysql.connector import Error
from paramiko.ssh_exception import SSHException
from dotenv import load_dotenv
import os
from concurrent.futures import ThreadPoolExecutor, as_completed

# Загружаем параметры из .env файла Laravel
load_dotenv('/path/to/your/laravel/.env')  # Укажите путь к вашему .env файлу Laravel

DB_HOST = os.getenv('DB_HOST')
DB_USER = os.getenv('DB_USERNAME')
DB_PASSWORD = os.getenv('DB_PASSWORD')
DB_NAME = os.getenv('DB_DATABASE')

# Подключение к базе данных Laravel
def get_db_connection():
    try:
        conn = mysql.connector.connect(
            host=DB_HOST,
            user=DB_USER,
            password=DB_PASSWORD,
            database=DB_NAME
        )
        if conn.is_connected():
            return conn
    except Error as e:
        print(f"Error connecting to MySQL: {e}")
        return None

# Функция для выполнения команды на узле через SSH
def execute_ssh_command(ip_address, username, password, command):
    try:
        ssh = paramiko.SSHClient()
        ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        ssh.connect(ip_address, username=username, password=password)

        stdin, stdout, stderr = ssh.exec_command(command)
        output = stdout.read().decode()
        error = stderr.read().decode()

        ssh.close()

        if error:
            return f"[{ip_address}] Ошибка: {error}"
        else:
            return f"[{ip_address}] Результат: {output}"

    except SSHException as e:
        return f"[{ip_address}] Не удалось подключиться: {e}"

# Функция для получения всех узлов из базы данных Laravel
def get_all_nodes():
    conn = get_db_connection()
    if not conn:
        return []

    cursor = conn.cursor()
    cursor.execute("SELECT ip_address, ssh_username, ssh_password, status FROM host_nodes")
    nodes = cursor.fetchall()
    conn.close()
    return nodes

# Основная функция для обработки аргументов и выполнения команды
def main():
    parser = argparse.ArgumentParser(description="Утилита для управления узлами кластера через SSH")
    parser.add_argument('-node', type=str, help="IP узла или 'all' для выполнения на всех узлах")
    parser.add_argument('-cmd', type=str, help="Команда для выполнения на узле")
    args = parser.parse_args()

    if args.node == 'all':
        # Получаем все узлы из базы данных и выполняем команду на каждом параллельно
        nodes = get_all_nodes()
        with ThreadPoolExecutor(max_workers=10) as executor:
            futures = []
            for node in nodes:
                ip_address, ssh_username, ssh_password, status = node
                if status == 'active':  # Пропускаем выключенные узлы
                    futures.append(executor.submit(execute_ssh_command, ip_address, ssh_username, ssh_password, args.cmd))
                else:
                    print(f"[{ip_address}] Узел выключен, пропуск...")

            # Выводим результаты по мере выполнения
            for future in as_completed(futures):
                print(future.result())

    else:
        # Выполняем команду на конкретном узле
        ip_address = args.node
        conn = get_db_connection()
        if not conn:
            print("Не удалось подключиться к базе данных.")
            return

        cursor = conn.cursor()
        cursor.execute("SELECT ssh_username, ssh_password, status FROM host_nodes WHERE ip_address = %s", (ip_address,))
        node = cursor.fetchone()
        conn.close()

        if node:
            ssh_username, ssh_password, status = node
            if status == 'active':
                result = execute_ssh_command(ip_address, ssh_username, ssh_password, args.cmd)
                print(result)
            else:
                print(f"[{ip_address}] Узел выключен, пропуск...")
        else:
            print(f"Узел с IP {ip_address} не найден.")

if __name__ == "__main__":
    main()
