<?php

namespace App\Services;

use App\Models\VirtualMachine;
use App\Models\VirtualMachineTemplate;

class VMService
{
    /**
     * Генерация XML-конфигурации для создания виртуальной машины
     *
     * @param VirtualMachineTemplate $template
     * @param VirtualMachine $vm
     * @return string
     */
    public function generateXmlConfig(VirtualMachineTemplate $template, VirtualMachine $vm)
    {
        $xml = new \SimpleXMLElement('<domain/>');
        $xml->addAttribute('type', 'kvm');

        $xml->addChild('name', $vm->vm_name);
        $memory = $xml->addChild('memory', $vm->memory * 1024);
        $memory->addAttribute('unit', 'KiB');
        $vcpu = $xml->addChild('vcpu', $vm->cpu);

        $cpu = $xml->addChild('cpu');
        $cpu->addAttribute('mode', $template->cpu_emulation);

        $devices = $xml->addChild('devices');
        $interface = $devices->addChild('interface');
        $interface->addAttribute('type', 'bridge');
        $interface->addChild('mac')->addAttribute('address', $this->generateMacAddress());
        $interface->addChild('source')->addAttribute('bridge', $template->network_bridge);
        $interface->addChild('model')->addAttribute('type', $template->network_model);

        $disk = $devices->addChild('disk');
        $disk->addAttribute('type', 'file');
        $disk->addAttribute('device', 'disk');
        $disk->addChild('driver')->addAttribute('type', $template->storagePool->format_disk);
        $disk->addChild('source')->addAttribute('file', $this->getDiskImagePath($vm));
        $disk->addChild('target')->addAttribute('dev', 'vda');
        $disk->addChild('target')->addAttribute('bus', 'virtio');

        // Добавление других параметров на основе шаблона
        // ...

        return $xml->asXML();
    }

    private function generateMacAddress()
    {
        return implode(':', array_map(function () {
            return sprintf('%02x', rand(0, 255));
        }, range(1, 6)));
    }

    private function getDiskImagePath(VirtualMachine $vm)
    {
        return '/var/lib/libvirt/images/' . $vm->vm_name . '.img';
    }
}
