<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\IpAddress;
use App\Models\HostNode;
use App\Models\VmStats;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class VirtualMachine extends Model
{
    use HasFactory;

    protected $fillable = [
        'node_id',
        'vm_name',
        'description',
        'status',
        'cpu',
        'memory',
        'disk_size',
        'migration_progress',
        'ssh_username',
        'ssh_password',
        'rdp_username',
        'rdp_password',
        'domain',
        'vnc_password',
        'cpu_emulation',
        'network_in',
        'network_out',
    ];


    public function node() : BelongsTo
    {
        return $this->belongsTo(HostNode::class);
    }

    public function ipAddresses() : HasMany
    {
        return $this->hasMany(IpAddress::class);
    }

    public function stats() : HasMany
    {
        return $this->hasMany(VmStats::class, 'vm_id');
    }

    public function hostNode() : BelongsTo
    {
        return $this->belongsTo(HostNode::class, 'node_id');
    }

    // Связь с ISO образами и дисками
    public function isoImages() : HasMany
    {
        return $this->hasMany(IsoImage::class);
    }

    public function disks() : HasMany
    {
        return $this->hasMany(Disk::class);
    }

    // Методы для обновления конфигурации
    public function updateConfiguration(array $data)
    {
        $this->update($data);
    }



    // Метод для назначения IP-адреса
    public function assignIpAddress($ipAddress, $type = 'ipv4')
    {
        if (!IpAddress::isAvailable($ipAddress)) {
            throw new \Exception('IP address is not available.');
        }

        IpAddress::create([
            'virtual_machine_id' => $this->id,
            'ip_address' => $ipAddress,
            'type' => $type,
            'status' => 'assigned',
        ]);
    }

    // Метод для освобождения IP-адреса
    public function releaseIpAddress($ipAddress)
    {
        $ip = IpAddress::where('ip_address', $ipAddress)
            ->where('virtual_machine_id', $this->id)
            ->first();

        if ($ip) {
            $ip->update(['status' => 'available', 'virtual_machine_id' => null]);
        }
    }

    public function allocateIpAddress()
    {
        $subnet = Subnet::where('type', $this->getIpType())->first();

        if ($subnet) {
            $ipAddress = $subnet->allocateIp();
            IpAddress::create([
                'subnet_id' => $subnet->id,
                'virtual_machine_id' => $this->id,
                'ip_address' => $ipAddress,
                'type' => $subnet->type,
                'status' => 'assigned',
            ]);
        } else {
            throw new \Exception('No suitable subnet found for IP allocation.');
        }
    }

    protected function getIpType()
    {
        // Метод, возвращающий тип IP (ipv4 или ipv6) на основе конфигурации или параметров VM
        return 'ipv4'; // Или 'ipv6', если это необходимо
    }

    //Резерв Ip IpAddress::reserveIpAddress('192.168.1.10');
    //Освобождение ip  IpAddress::releaseReservedIpAddress('192.168.1.10');
    //назначение  $vm = VirtualMachine::find($vmId);
    //$ipv4Addresses = ['192.168.1.10', '192.168.1.11'];
    //$ipv6Addresses = ['2001:db8::1', '2001:db8::2'];
    //IpAddress::assignIpAddresses($vm, $ipv4Addresses, $ipv6Addresses);

    //Освобождение ip по виртуалкам $vm = VirtualMachine::find($vmId);
    //IpAddress::releaseIpAddresses($vm);

//$user = auth()->user();
//
//    // Проверяем, имеет ли пользователь права на создание виртуальных машин на технических нодах
//if ($this->isCreatingOnTechnicalNode($request->input('node_id')) && !$user->can('create_on_technical_nodes')) {
//return response()->json(['error' => 'Недостаточно прав для создания виртуальных машин на технической ноде.'], 403);
//}
}
