<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\DB;
use App\Models\IpAddress;

class Subnet extends Model
{
    use HasFactory;

    protected $fillable = [
        'subnet',
        'total_ips',
        'type', // ipv4 или ipv6
        'is_ip_list', // Если true, это не подсеть, а список IP
    ];

    public function ipAddresses() : HasMany
    {
        return $this->hasMany(IpAddress::class);
    }

    public function countAvailableIps()
    {
        return $this->ipAddresses()->where('status', 'available')->count();
    }

    public function allocateIp()
    {
        if ($this->available_ips > 0) {
            $ipAddress = $this->getNextAvailableIp();
            $this->decrement('available_ips');
            return $ipAddress;
        }

        throw new \Exception('No available IP addresses in this subnet.');
    }

    public function getNextAvailableIp()
    {
        $usedIps = $this->ipAddresses->pluck('ip_address')->toArray();

        if ($this->type === 'ipv4') {
            return $this->getNextAvailableIpv4($this->subnet, $usedIps);
        } else {
            return $this->getNextAvailableIpv6($this->subnet, $usedIps);
        }
    }

    private function getNextAvailableIpv4($subnet, $usedIps)
    {
        $subnetParts = explode('/', $subnet);
        $network = $subnetParts[0];
        $cidr = isset($subnetParts[1]) ? (int)$subnetParts[1] : 32;

        $networkLong = ip2long($network);
        $mask = ~((1 << (32 - $cidr)) - 1);
        $startIpLong = $networkLong & $mask;
        $endIpLong = $startIpLong | (~$mask);

        for ($ipLong = $startIpLong + 1; $ipLong < $endIpLong; $ipLong++) {
            $ip = long2ip($ipLong);
            if (!in_array($ip, $usedIps)) {
                return $ip;
            }
        }

        throw new \Exception('No available IP addresses in this subnet.');
    }

    private function getNextAvailableIpv6($subnet, $usedIps)
    {
        // Использования GMP для работы с IPv6
        list($network, $cidr) = explode('/', $subnet);
        $network = inet_pton($network); // Преобразование в двоичный формат
        $cidr = (int) $cidr;

        $startIp = gmp_init(bin2hex($network), 16);
        $networkSize = gmp_sub(gmp_pow(2, 128 - $cidr), 1);
        $endIp = gmp_add($startIp, $networkSize);

        $currentIp = gmp_add($startIp, 1); // Начинаем после адреса сети
        while (gmp_cmp($currentIp, $endIp) < 0) {
            $ip = inet_ntop(hex2bin(gmp_strval($currentIp, 16)));
            if (!in_array($ip, $usedIps)) {
                return $ip;
            }
            $currentIp = gmp_add($currentIp, 1);
        }

        throw new \Exception('No available IP addresses in this subnet.');
    }

    public static function calculateTotalIps($subnet)
    {
        list($network, $cidr) = explode('/', $subnet);
        if (filter_var($network, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            return pow(2, 32 - (int)$cidr);
        } elseif (filter_var($network, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
            return gmp_strval(gmp_pow(2, 128 - (int)$cidr));
        }
        throw new \Exception('Invalid subnet format.');
    }

}
