<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use App\Models\VirtualMachine;
use App\Models\Subnet;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class IpAddress extends Model
{
    use HasFactory;

    protected $fillable = [
        'subnet_id',
        'virtual_machine_id',
        'ip_address',
        'type',
        'status',
        'comment',
    ];

    public function virtualMachine() : BelongsTo
    {
        return $this->belongsTo(VirtualMachine::class);
    }

    public function subnet() : BelongsTo
    {
        return $this->belongsTo(Subnet::class);
    }

    public static function isAvailable($ipAddress)
    {
        return !self::where('ip_address', $ipAddress)
            ->where('status', 'reserved')
            ->exists();
    }

    public static function reserveIpAddress($ipAddress)
    {
        $ip = self::where('ip_address', $ipAddress)->first();
        if ($ip) {
            $ip->update(['status' => 'reserved']);
        } else {
            throw new \Exception('IP address not found.');
        }
    }

    public static function releaseReservedIpAddress($ipAddress)
    {
        $ip = self::where('ip_address', $ipAddress)->first();
        if ($ip) {
            $ip->update(['status' => 'available']);
        } else {
            throw new \Exception('IP address not found.');
        }
    }

    public function assignIpAddresses($vm, array $ipv4Addresses, array $ipv6Addresses)
    {
        foreach ($ipv4Addresses as $ip) {
            self::updateOrCreate(
                ['ip_address' => $ip],
                ['virtual_machine_id' => $vm->id, 'type' => 'ipv4', 'status' => 'assigned']
            );
        }

        foreach ($ipv6Addresses as $ip) {
            self::updateOrCreate(
                ['ip_address' => $ip],
                ['virtual_machine_id' => $vm->id, 'type' => 'ipv6', 'status' => 'assigned']
            );
        }
    }

    public function releaseIpAddresses($vm)
    {
        self::where('virtual_machine_id', $vm->id)
            ->update(['status' => 'available', 'virtual_machine_id' => null]);
    }
}
