<?php

namespace App\Jobs;

use App\Models\Node;
use App\Models\Subnet;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class SyncSubnetToNodes implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $subnet;
    protected $isNewNode;

    /**
     * Create a new job instance.
     *
     * @param Subnet $subnet
     * @param bool $isNewNode
     * @return void
     */
    public function __construct(Subnet $subnet, $isNewNode = false)
    {
        $this->subnet = $subnet;
        $this->isNewNode = $isNewNode; // Указывает, новый ли узел, чтобы не пропускать подсети
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // Получаем все узлы
        $nodes = Node::all();

        foreach ($nodes as $node) {
            // Проверяем, существует ли подсеть на узле или если это новый узел
            if (!$this->checkSubnetExistsOnNode($node, $this->subnet) || $this->isNewNode) {
                // Настраиваем подсеть на узле через SSH или API
                $this->configureSubnetOnNode($node, $this->subnet);
            }
        }
    }

    /**
     * Проверяет, существует ли подсеть на узле
     *
     * @param Node $node
     * @param Subnet $subnet
     * @return bool
     */
    protected function checkSubnetExistsOnNode(Node $node, Subnet $subnet)
    {
        // Логика для проверки, существует ли подсеть на узле.
        // Например, можно выполнить команду на узле через SSH и проверить конфигурацию сети.
        $sshCommand = "ip route show | grep '{$subnet->cidr}'";

        // Выполняем команду SSH для проверки наличия подсети
        $output = $this->executeSshCommand($node, $sshCommand);

        // Возвращаем true, если подсеть уже существует
        return !empty($output);
    }

    /**
     * Настраивает подсеть на узле
     *
     * @param Node $node
     * @param Subnet $subnet
     * @return void
     */
    protected function configureSubnetOnNode(Node $node, Subnet $subnet)
    {
        // Логика для настройки подсети на узле. Например, можно отправить SSH-команду для добавления маршрута.
        $sshCommand = "ip route add {$subnet->cidr} dev {$node->network_interface}";

        // Выполняем команду SSH для настройки подсети на узле
        $this->executeSshCommand($node, $sshCommand);

        // Логируем успешную настройку подсети на узле
        \Log::info("Подсеть {$subnet->cidr} добавлена на узел {$node->name}.");
    }

    /**
     * Выполняет SSH команду на узле
     *
     * @param Node $node
     * @param string $command
     * @return string
     */
    protected function executeSshCommand(Node $node, string $command)
    {
        // Пример выполнения команды через SSH:
        try {
            $sshConnection = ssh2_connect($node->ip_address, 22);
            ssh2_auth_password($sshConnection, $node->username, $node->password);
            $stream = ssh2_exec($sshConnection, $command);
            stream_set_blocking($stream, true);
            $output = stream_get_contents($stream);
            fclose($stream);
            return $output;
        } catch (\Exception $e) {
            \Log::error("Ошибка при выполнении SSH команды на узле {$node->name}: {$e->getMessage()}");
            return '';
        }
    }
}
