<?php

namespace App\Jobs;

use App\Models\VirtualMachine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;

class OfflineMigrateVirtualMachine implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $virtualMachine;
    protected $targetNode;

    /**
     * Create a new job instance.
     */
    public function __construct(VirtualMachine $virtualMachine, $targetNode)
    {
        $this->virtualMachine = $virtualMachine;
        $this->targetNode = $targetNode;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        $sourceNode = $this->virtualMachine->node;

        // Остановить ВМ на источнике
        $this->sendStopRequest($sourceNode);

        // Запустить ВМ на новом узле
        $this->sendCreateRequest($this->targetNode);

        // Удалить ВМ с исходного узла
        $this->sendDeleteRequest($sourceNode);

        // Обновляем узел ВМ в базе данных
        $this->virtualMachine->update(['node_id' => $this->targetNode->id]);
    }

    protected function sendStopRequest($node)
    {
        $url = "http://{$node->ip_address}:{$node->agent_port}/server/stop";
        Http::post($url, ['name' => $this->virtualMachine->vm_name]);
    }

    protected function sendCreateRequest($node)
    {
        $url = "http://{$node->ip_address}:{$node->agent_port}/server/create";
        Http::post($url, [
            'name' => $this->virtualMachine->vm_name,
            'cpu' => $this->virtualMachine->cpu,
            'memory' => $this->virtualMachine->memory,
            'disk_size' => $this->virtualMachine->disk_size,
            'network_bridge' => 'br0',
            'network_model' => 'virtio',
        ]);
    }

    protected function sendDeleteRequest($node)
    {
        $url = "http://{$node->ip_address}:{$node->agent_port}/server/delete";
        Http::post($url, ['name' => $this->virtualMachine->vm_name]);
    }
}
