<?php

namespace App\Jobs;

use App\Models\HostNode;
use App\Models\JobMonitoring;
use App\Models\Repository;
use App\Models\Pool; // Добавлено для работы с пулом
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use phpseclib3\Net\SSH2;
use Illuminate\Support\Facades\Log;

class InstallSoftwareOnNode implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $hostNode;
    protected $pool; // Добавляем объект пула

    /**
     * Create a new job instance.
     *
     * @param HostNode $hostNode
     */
    public function __construct(HostNode $hostNode)
    {
        $this->hostNode = $hostNode;
        $this->pool = Pool::where('host_node_id', $hostNode->id)->first(); // Получаем первый пул, привязанный к узлу
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // Логируем начало задачи
        Log::channel('node_logs')->info('Начата установка ПО на ноде: ' . $this->hostNode->name);

        // Создаём запись в мониторинге задач
        $jobMonitoring = JobMonitoring::create([
            'job_name' => 'InstallSoftwareOnNode',
            'status' => 'started',
            'details' => 'Начата установка ПО на ноду: ' . $this->hostNode->name,
        ]);

        // Обновляем статус ноды на 'installing'
        $this->hostNode->update(['status' => 'installing']);

        // Подключаемся к ноде по SSH
        $ssh = new SSH2($this->hostNode->ip_address);
        if (!$ssh->login($this->hostNode->ssh_username, $this->hostNode->ssh_password)) {
            Log::channel('node_logs')->error('Не удалось подключиться по SSH к ноде: ' . $this->hostNode->name);
            $this->hostNode->update(['status' => 'inactive']);
            $jobMonitoring->update([
                'status' => 'failed',
                'details' => 'Не удалось подключиться по SSH к ноде: ' . $this->hostNode->name,
            ]);
            return;
        }

        Log::channel('node_logs')->info('Подключение по SSH к ноде успешно: ' . $this->hostNode->name);

        // Выполняем команды для установки необходимого ПО на узле
        $commands = [
            'sudo dnf update -y',
            'sudo dnf install -y epel-release',
            'sudo dnf install -y qemu-kvm libvirt virt-install bridge-utils nfs-utils libvirt-devel',
            'sudo systemctl enable --now libvirtd',
            'sudo systemctl start virtlogd',
            'sudo dnf install -y tar',
            'sudo usermod -aG libvirt $(whoami)',
            // Установка Python, если не установлен, проверка версии
            'if ! command -v python3 &>/dev/null; then sudo dnf install -y python3; fi',
            'PYTHON_VERSION=$(python3 --version 2>&1 | grep -oP "(?<=Python )\d+\.\d+")',
            'if [[ "$PYTHON_VERSION" < "3.8" ]]; then sudo dnf install -y python38; fi',
            'sudo pip3 install psutil requests',
            'sudo dnf install -y libvirt-python',
        ];

        if ($this->pool && $this->pool->type === 'lvm') {
            $diskDevice = $this->pool->disk_device ?? '/dev/sdX'; // Используем устройство, указанное в пуле

            Log::channel('node_logs')->info('Настройка LVM на ноде: ' . $this->hostNode->name);

            $commands = array_merge($commands, [
                'sudo dnf install -y lvm2',
                'sudo pvcreate ' . $diskDevice,  // Используем устройство из формы
                'sudo vgcreate ' . $this->pool->name . ' ' . $diskDevice,  // Создание volume group
                'sudo lvcreate -L ' . $this->pool->reserve . 'G -n ' . $this->pool->name . '_lv ' . $this->pool->name,  // Создание logical volume
                'sudo mkfs.ext4 /dev/' . $this->pool->name . '/' . $this->pool->name . '_lv',  // Форматирование тома
            ]);

            Log::channel('node_logs')->info('LVM успешно настроен на ноде: ' . $this->hostNode->name);
        }
        foreach ($commands as $command) {
            $ssh->exec($command);
            Log::channel('node_logs')->info('Выполнена команда: ' . $command . ' на ноде: ' . $this->hostNode->name);
        }

        // Проверка статуса службы libvirtd
        $libvirtStatus = $ssh->exec('systemctl is-active libvirtd');
        $virtlogdStatus = $ssh->exec('systemctl is-active virtlogd');

        if (trim($libvirtStatus) === 'active' && trim($virtlogdStatus) === 'active') {
            Log::channel('node_logs')->info('libvirtd и virtlogd успешно запущены на ноде: ' . $this->hostNode->name);

            // Остальные действия по установке агентов
            // ...

            $this->hostNode->update(['status' => 'active', 'last_checked_at' => now()]);
            $jobMonitoring->update([
                'status' => 'completed',
                'details' => 'Установка завершена на ноде: ' . $this->hostNode->name,
            ]);
        } else {
            Log::channel('node_logs')->error('Ошибка при установке libvirtd или virtlogd на ноде: ' . $this->hostNode->name);
            $this->hostNode->update(['status' => 'inactive']);
            $jobMonitoring->update([
                'status' => 'failed',
                'details' => 'Ошибка при установке libvirtd или virtlogd на ноде: ' . $this->hostNode->name,
            ]);
        }

        $ssh->disconnect();
        Log::channel('node_logs')->info('SSH соединение закрыто для ноды: ' . $this->hostNode->name);
    }
}
