<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Exception;
use Illuminate\Support\Facades\Log;

class DownloadAndExtractOsArchive implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $archiveName;
    protected $downloadUrl;

    /**
     * Create a new job instance.
     *
     * @param string $archiveName - Имя архива (например, os-image.tar.gz)
     * @param string $downloadUrl - URL для загрузки архива
     */
    public function __construct($archiveName, $downloadUrl)
    {
        $this->archiveName = $archiveName;
        $this->downloadUrl = $downloadUrl;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // Путь к директории nffshare, куда нужно разархивировать
        $nffsharePath = '/nffshare';

        try {
            // Шаг 1: Загрузка архива
            $this->downloadArchive($nffsharePath);

            // Шаг 2: Разархивирование в директорию nffshare/<archive_name_without_extension>
            $this->extractArchive($nffsharePath);

            // Шаг 3: Удаление архива после распаковки
            $this->deleteArchive($nffsharePath);

        } catch (Exception $e) {
            // Логируем ошибку, если произошел сбой
            Log::error("Ошибка при загрузке или разархивировании архива {$this->archiveName}: {$e->getMessage()}");
        }
    }

    /**
     * Метод для загрузки архива из репозитория
     *
     * @param string $path
     * @return void
     */
    protected function downloadArchive($path)
    {
        // Полный путь для сохранения архива
        $archivePath = $path . '/' . $this->archiveName;

        // HTTP-запрос на загрузку архива
        $response = Http::get($this->downloadUrl);

        // Проверяем, успешен ли запрос
        if ($response->successful()) {
            // Сохраняем архив в директории nffshare
            file_put_contents($archivePath, $response->body());
        } else {
            throw new Exception("Не удалось загрузить архив {$this->archiveName}");
        }
    }

    /**
     * Метод для разархивирования архива в нужную директорию
     *
     * @param string $path
     * @return void
     */
    protected function extractArchive($path)
    {
        // Путь к загруженному архиву
        $archivePath = $path . '/' . $this->archiveName;

        // Проверяем, существует ли архив
        if (!file_exists($archivePath)) {
            throw new Exception("Архив {$this->archiveName} не найден для разархивирования.");
        }

        // Папка для распаковки, её название будет совпадать с именем архива без .tar.gz
        $extractPath = $path . '/' . pathinfo($this->archiveName, PATHINFO_FILENAME);

        // Разархивируем tar.gz
        $command = "mkdir -p {$extractPath} && tar -xzf {$archivePath} -C {$extractPath}";
        $output = shell_exec($command);

        if ($output === null) {
            throw new Exception("Ошибка при разархивировании архива {$this->archiveName}.");
        }

        // Проверяем, что внутри архива
        $files = scandir($extractPath);
        $imageFile = null;
        $cloudInitFile = null;

        foreach ($files as $file) {
            if (preg_match('/\.(img|qcow2)$/', $file)) {
                $imageFile = $file;
            } elseif ($file === 'cloud-init.yaml') {
                $cloudInitFile = $file;
            }
        }

        if (!$imageFile) {
            throw new Exception("Не найдено подходящего образа (img или qcow2) в архиве.");
        }

        if (!$cloudInitFile) {
            throw new Exception("Файл cloud-init.yaml отсутствует в архиве.");
        }

        // Возвращаем путь к извлеченным файлам
        return [
            'image' => $extractPath . '/' . $imageFile,
            'cloud_init' => $extractPath . '/' . $cloudInitFile
        ];
    }


    /**
     * Метод для удаления архива после успешного разархивирования
     *
     * @param string $path
     * @return void
     */
    protected function deleteArchive($path)
    {
        $archivePath = $path . '/' . $this->archiveName;

        // Удаление архива
        if (file_exists($archivePath)) {
            unlink($archivePath);
        } else {
            throw new Exception("Не удалось найти архив {$this->archiveName} для удаления.");
        }
    }
}
