<?php

namespace App\Jobs;

use App\Models\HostNode;
use App\Models\VirtualMachine;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class DeleteVirtualMachineJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $hostNode;
    protected $vm;

    /**
     * Create a new job instance.
     *
     * @param HostNode $hostNode
     * @param VirtualMachine $vm
     */
    public function __construct(HostNode $hostNode, VirtualMachine $vm)
    {
        $this->hostNode = $hostNode;
        $this->vm = $vm;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // Удаление виртуальной машины с хост-ноды
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->hostNode->auth_key,
        ])->post("http://{$this->hostNode->ip_address}:1515/server/delete", [
            'name' => $this->vm->vm_name,
        ]);

        if ($response->successful()) {
            // Если удаление на хост-ноде успешно, удаляем диски и сетевые интерфейсы
            $this->deleteAssociatedResources();

            // Успешное удаление виртуальной машины
            $this->vm->update(['status' => 'deleted']);
            Log::info("Virtual machine '{$this->vm->vm_name}' deleted from node '{$this->hostNode->name}'");
        } else {
            // Если ошибка при удалении на хост-ноде
            $this->vm->update(['status' => 'error']);
            Log::error("Failed to delete virtual machine '{$this->vm->vm_name}' from node '{$this->hostNode->name}': {$response->body()}");
        }
    }

    /**
     * Удаление связанных ресурсов: дисков и сетевых интерфейсов.
     */
    private function deleteAssociatedResources()
    {
        // Удаляем диски
        foreach ($this->vm->disks as $disk) {
            $disk->delete();
        }

        // Удаляем сетевые интерфейсы
        foreach ($this->vm->networkInterfaces as $interface) {
            $interface->delete();
        }

        Log::info("Associated resources for VM '{$this->vm->vm_name}' deleted: disks and network interfaces.");
    }
}
