<?php

namespace App\Jobs;

use App\Models\HostNode;
use App\Models\VirtualMachineTemplate;
use App\Models\IpAddress;
use App\Models\Subnet;
use App\Models\Disk;
use App\Models\NetworkInterfaces;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use phpseclib3\Net\SSH2;
use Exception;

class CreateVirtualMachineJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $hostNode;
    protected $template;
    protected $vmName;

    public function __construct(HostNode $hostNode, VirtualMachineTemplate $template, $vmName)
    {
        $this->hostNode = $hostNode;
        $this->template = $template;
        $this->vmName = $vmName;
    }

    public function handle()
    {
        $ssh = new SSH2($this->hostNode->ip_address);
        if (!$ssh->login($this->hostNode->ssh_username, $this->hostNode->ssh_password)) {
            Log::error('Не удалось подключиться по SSH к ноде: ' . $this->hostNode->name);
            $this->template->update(['status' => 'error']);
            return;
        }

        try {
            $subnet = Subnet::where('type', 'ipv4')->firstOrFail();
            $ipAddress = $subnet->getNextAvailableIp();
            IpAddress::reserveIpAddress($ipAddress);

            $macAddress = NetworkInterfaces::generateMacAddress();
            $diskPath = $this->template->storagePool->path;

            // Получаем формат диска из модели Pool
            $formatDisk = $this->template->storagePool->format_disk;

            $command = sprintf(
                '/usr/local/bin/vm_agent create --name %s --memory %d --vcpu %d --disk_size %d --iso %s --ip %s --mac %s --disk_path %s --cpu_emulation %s --cpu_weight %d --io_weight %d --network_in %d --network_out %d --format_disk %s',
                escapeshellarg($this->vmName),
                $this->template->memory,
                $this->template->cpu,
                $this->template->disk_size,
                escapeshellarg($this->template->os_directory),
                escapeshellarg($ipAddress),
                escapeshellarg($macAddress),
                escapeshellarg($diskPath),
                escapeshellarg($this->template->cpu_emulation),
                $this->template->cpu_weight,
                $this->template->io_weight,
                $this->template->network_in,
                $this->template->network_out,
                escapeshellarg($formatDisk)  // Передаем формат диска
            );

            $result = $ssh->exec($command);

            if (strpos($result, 'created successfully') !== false) {
                Log::info("Виртуальная машина '{$this->vmName}' создана на ноде '{$this->hostNode->name}'");

                $this->template->update(['status' => 'installing']);

                $this->waitForInstallation($ssh);

                $this->template->update(['status' => 'ready']);

                Disk::create([
                    'virtual_machine_id' => $this->template->id,
                    'host_node_id' => $this->hostNode->id,
                    'size' => $this->template->disk_size,
                    'path' => $diskPath . '/' . $this->vmName . '_disk_' . uniqid() . '.' . $this->template->storagePool->format_disk,
                    'type' => $this->template->storagePool->format_disk,
                    'status' => 'created',
                ]);

                NetworkInterfaces::create([
                    'mac_address' => $macAddress,
                    'virtual_machine_id' => $this->template->id,
                ]);

                IpAddress::create([
                    'subnet_id' => $subnet->id,
                    'virtual_machine_id' => $this->template->id,
                    'ip_address' => $ipAddress,
                    'type' => 'ipv4',
                    'status' => 'assigned',
                ]);

            } else {
                throw new Exception("Не удалось создать виртуальную машину '{$this->vmName}' на ноде '{$this->hostNode->name}': $result");
            }
        } catch (Exception $e) {
            Log::error($e->getMessage());
            $this->template->update(['status' => 'error']);
        }

        $ssh->disconnect();
    }

    private function waitForInstallation($ssh)
    {
        while (true) {
            $statusCheckCommand = '/usr/local/bin/vm_agent check_installation_status --name ' . escapeshellarg($this->vmName);
            $status = $ssh->exec($statusCheckCommand);

            if (strpos($status, 'OS installation completed') !== false) {
                Log::info("Установка ОС завершена для виртуальной машины '{$this->vmName}'");
                break;
            }

            sleep(60);
        }
    }
}
