<?php

namespace App\Jobs;

use App\Models\HostNode;
use App\Notifications\NodeUnavailableNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use phpseclib3\Net\SSH2;

class CheckNodeAvailability implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $hostNode;

    /**
     * Create a new job instance.
     *
     * @param HostNode $hostNode
     */
    public function __construct(HostNode $hostNode)
    {
        $this->hostNode = $hostNode;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        // Проверяем статус ноды, если она считается выключенной, то пропускаем проверку
        if ($this->hostNode->status === 'inactive') {
            return;
        }

        // Проверка доступности по ping
        $pingResult = shell_exec(sprintf('ping -c 1 %s', escapeshellarg($this->hostNode->ip_address)));
        $pingSuccess = strpos($pingResult, '1 received') !== false;

        // Проверка доступности по SSH
        $ssh = new SSH2($this->hostNode->ip_address);
        $sshSuccess = $ssh->login($this->hostNode->ssh_username, $this->hostNode->ssh_password);

        if ($pingSuccess && $sshSuccess) {
            $this->hostNode->update(['is_available' => true, 'last_checked_at' => now()]);
            Log::info('Node ' . $this->hostNode->name . ' is available.');
        } else {
            $this->hostNode->update(['is_available' => false, 'last_checked_at' => now()]);
            $this->hostNode->notify(new NodeUnavailableNotification());
            Log::warning('Node ' . $this->hostNode->name . ' is unavailable.');
        }

        // Можно добавить уведомление в панель о недоступности ноды, например, через внутренние уведомления
    }
}

