<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Role;

class InstallerController extends Controller
{
    public function index()
    {
        return view('install.index');
    }

    public function install(Request $request)
    {
        // Валидация данных
        $request->validate([
            'admin_name' => 'required|string|max:255',
            'admin_email' => 'required|string|email|max:255|unique:users,email',
            'admin_password' => 'required|string|min:8|confirmed',
        ]);

        // Проверка, что все необходимые пакеты установлены
        if (!extension_loaded('pdo_mysql')) {
            return redirect()->back()->withErrors('MySQL PDO расширение не установлено.');
        }
        if (!extension_loaded('redis')) {
            return redirect()->back()->withErrors('Redis расширение не установлено.');
        }

        try {
            // Выполнение миграций и сидеров
            Artisan::call('migrate', ['--force' => true]);
            Artisan::call('db:seed', ['--class' => 'PermissionsTableSeeder', '--force' => true]);
            Artisan::call('db:seed', ['--class' => 'RolePermissionSeeder', '--force' => true]);

            // Создание администратора
            $adminRole = Role::where('name', 'admin')->first();

            $admin = User::create([
                'name' => $request->admin_name,
                'email' => $request->admin_email,
                'password' => Hash::make($request->admin_password),
            ]);

            // Присваиваем роль администратора
            $admin->assignRole($adminRole);

            return redirect()->route('login')->with('success', 'Установка завершена. Вы можете войти в систему.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors('Ошибка во время установки: ' . $e->getMessage());
        }
    }
}
