<?php

namespace App\Http\Controllers;

use App\Jobs\CreateVirtualMachineJob;
use App\Jobs\DeleteVirtualMachineJob;
use App\Jobs\MigrateVirtualMachine;
use App\Jobs\StartVirtualMachineJob;
use App\Jobs\StopVirtualMachineJob;
use App\Models\IpAddress;
use App\Models\NetworkInterfaces;
use App\Models\Pool;
use App\Models\Repository;
use App\Models\VirtualMachine;
use App\Models\HostNode;


use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class VirtualMachineController extends Controller
{
//    public function __construct()
//    {
//        $this->middleware('can:view_virtual_machines')->only(['index', 'show']);
//        $this->middleware('can:create_virtual_machines')->only(['create', 'store']);
//        $this->middleware('can:edit_virtual_machines')->only(['edit', 'update']);
//        $this->middleware('can:delete_virtual_machines')->only(['destroy']);
//    }

    public function index(Request $request)
    {
        $query = VirtualMachine::query();

        if ($request->has('search')) {
            $search = $request->input('search');
            $query->where('vm_name', 'LIKE', "%{$search}%")
                ->orWhereHas('ipAddresses', function($q) use ($search) {
                    $q->where('ip_address', 'LIKE', "%{$search}%");
                });
        }

        $virtualMachines = $query->get();

        return view('virtual_machines.index', compact('virtualMachines'));
    }

    public function create()
    {
        $user = Auth::user();

        // Проверка роли пользователя
        if ($user->hasRole('admin')) {
            // Если администратор, показываем все узлы
            $nodes = HostNode::all();
        } else {
            // Если не администратор, показываем только активные, не заблокированные и не технические узлы
            $nodes = HostNode::where('status', 'active')
                ->where('is_technical', false)
                ->where('is_locked', false)
                ->get();
        }

        // Проверка доступных IP-адресов
        $availableIps = IpAddress::where('status', 'available')->get();

        if ($nodes->isEmpty() || $availableIps->isEmpty()) {
            return redirect()->route('virtual_machines.index')->with('error', __('messages.no_active_nodes_or_ips'));
        }

        // Логика для выбора лучшего узла на основе свободных ресурсов
        $bestNode = $nodes->sortByDesc(function ($node) {
            $cpuWeight = $node->total_memory - $node->free_memory; // Пример веса по CPU
            $memoryWeight = $node->free_memory;
            $diskWeight = $node->free_disk * 1024; // Приводим ГБ к МБ для соотношения

            return $memoryWeight + $diskWeight - $cpuWeight;
        })->first();

        // Получаем доступные пулы для создания дисков
        $pools = Pool::all();

        // Получаем список сетевых моделей (или можно сделать динамическим)
        $networkModels = ['virtio', 'e1000', 'rtl8139'];

        // Генерация MAC-адреса
        $generatedMacAddress = $this->generateMacAddress();

        return view('virtual_machines.create', compact('nodes', 'bestNode', 'availableIps', 'pools', 'generatedMacAddress', 'networkModels'));
    }


    /**
     * Метод для генерации случайного MAC-адреса.
     */
    private function generateMacAddress()
    {
        return strtoupper(
            '02:00:00:' . dechex(rand(0, 255)) . ':' . dechex(rand(0, 255)) . ':' . dechex(rand(0, 255))
        );
    }


    public function store(Request $request)
    {
        // Валидация входных данных
        $request->validate([
            'vm_name' => 'required|string|unique:virtual_machines',
            'os_name' => 'required|string',
            'node_id' => 'required|exists:host_nodes,id',
            'cpu' => 'required|integer|min:1',
            'memory' => 'required|integer|min:1',
            'disks' => 'required|array',
            'disks.*.size' => 'required|integer|min:1', // размер каждого диска
            'disks.*.pool_id' => 'required|exists:pools,id', // пул для каждого диска
            'network_interfaces' => 'required|array',
            'network_interfaces.*.mac_address' => 'nullable|string', // необязательный MAC-адрес, если не передан, сгенерируем
            'network_interfaces.*.network_model' => 'required|string', // тип сетевого интерфейса
        ]);

        $repository = Repository::find(1);
        $osList = $repository->getOsListFromInfoJson();
        $selectedOs = collect($osList['operating_systems'])->firstWhere('name', $request->input('os_name'));

        if (!$selectedOs) {
            return redirect()->back()->with('error', 'Операционная система не найдена.');
        }

        // Создаём виртуальную машину
        $vm = VirtualMachine::create([
            'vm_name' => $request->input('vm_name'),
            'node_id' => $request->input('node_id'),
            'cpu' => $request->input('cpu'),
            'memory' => $request->input('memory'),
            'disk_size' => array_sum(array_column($request->input('disks'), 'size')), // общее количество памяти
            'os_file' => $selectedOs['file'],  // Добавляем выбранный файл ОС
            'method' => $selectedOs['method'], // Метод установки
        ]);

        // Обрабатываем диски
        foreach ($request->input('disks') as $diskData) {
            $pool = Pool::findOrFail($diskData['pool_id']); // получаем пул для каждого диска

            $vm->disks()->create([
                'size' => $diskData['size'],
                'type' => $pool->format_disk, // формат берется из пула
                'path' => $pool->path . '/' . $vm->vm_name . '_disk_' . uniqid() . '.' . $pool->format_disk, // уникальный путь диска
                'host_node_id' => $vm->node_id, // привязываем диск к ноде
                'status' => 'active',
            ]);
        }

        // Обрабатываем сетевые интерфейсы
        foreach ($request->input('network_interfaces') as $interfaceData) {
            $macAddress = $interfaceData['mac_address'] ?? NetworkInterfaces::generateMacAddress(); // генерируем MAC, если не задан

            $vm->networkInterfaces()->create([
                'mac_address' => $macAddress,
                'network_model' => $interfaceData['network_model'],
            ]);
        }

        // Запуск задания на создание виртуальной машины на хосте
        CreateVirtualMachineJob::dispatch($vm->node, $vm);

        return redirect()->route('virtual_machines.index')->with('success', __('messages.vm_created'));
    }


    public function edit(VirtualMachine $virtualMachine)
    {
        return view('virtual_machines.edit', compact('virtualMachine'));
    }

    public function update(Request $request, VirtualMachine $virtualMachine)
    {
        $request->validate([
            'vm_name' => 'required|string|unique:virtual_machines,vm_name,' . $virtualMachine->id,
            'cpu' => 'required|integer|min:1',
            'memory' => 'required|integer|min:1',
            'disk_size' => 'required|integer|min:1',
        ]);

        $virtualMachine->update($request->all());

        return redirect()->route('virtual_machines.index')->with('success', __('messages.vm_updated'));
    }

    public function destroy(VirtualMachine $virtualMachine)
    {
        $virtualMachine->delete();
        return redirect()->route('virtual_machines.index')->with('success', __('messages.vm_deleted'));
    }

    public function migrate(Request $request, $id)
    {
        $virtualMachine = VirtualMachine::findOrFail($id);
        $targetNode = HostNode::findOrFail($request->input('target_node_id'));

        // Запускаем job на миграцию
        MigrateVirtualMachine::dispatch($virtualMachine, $targetNode);

        return redirect()->back()->with('success', 'Миграция виртуальной машины запущена.');
    }

    public function bulkAction(Request $request)
    {
        $action = $request->input('action');
        $selectedVms = $request->input('selected_vms');

        if (!$selectedVms) {
            return redirect()->back()->with('error', __('messages.no_vms_selected'));
        }

        foreach ($selectedVms as $vmId) {
            $vm = VirtualMachine::find($vmId);

            switch ($action) {
                case 'migrate':
                    // Для миграции можно перенаправить на страницу с выбором узла или сразу использовать Job
                    $targetNodeId = $request->input('target_node_id'); // Получаем ID целевого узла
                    if ($targetNodeId) {
                        $targetNode = HostNode::find($targetNodeId);
                        MigrateVirtualMachine::dispatch($vm, $targetNode); // Запускаем Job миграции
                    } else {
                        return redirect()->back()->with('error', __('messages.select_target_node'));
                    }
                    break;

                case 'stop':
                    // Остановка виртуальной машины через Job
                    StopVirtualMachineJob::dispatch($vm->node, $vm->vm_name);
                    break;

                case 'start':
                    // Запуск виртуальной машины через Job
                    StartVirtualMachineJob::dispatch($vm->node, $vm->vm_name);
                    break;

                case 'delete':
                    // Удаление виртуальной машины через Job
                    DeleteVirtualMachineJob::dispatch($vm->node, $vm->vm_name);
                    break;

                default:
                    return redirect()->back()->with('error', __('messages.invalid_action'));
            }
        }

        return redirect()->back()->with('success', __('messages.action_completed'));
    }


}
