<?php

namespace App\Http\Controllers;

use App\Models\Repository;
use App\Models\TemplateOS;
use App\Jobs\DownloadAndInstallOs;
use App\Jobs\DeleteOsJob;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;

class TemplateOsController extends Controller
{

//    public function __construct()
//    {
//        $this->middleware('can:view_template_os')->only(['index', 'refresh']);
//        $this->middleware('can:download_template_os')->only(['download']);
//        $this->middleware('can:reinstall_template_os')->only(['reinstall']);
//        $this->middleware('can:delete_template_os')->only(['delete']);
//    }

    // Отображение списка ОС
    public function index()
    {
        // Получаем список всех установленных ОС
        $templateOsList = TemplateOS::all();

        // Создаем список ОС из репозиториев
        $osListFromRepositories = [];
        $repositories = Repository::all();

        foreach ($repositories as $repository) {
            try {
                // Пробуем получить данные из info.json
                $response = Http::get($repository->url . '/info.json');
                $osListFromRepositories[] = $response->json(); // Добавляем данные из каждого репозитория в общий список
            } catch (\Exception $e) {
                // Если произошла ошибка, добавляем сообщение об ошибке
                $osListFromRepositories[] = [
                    'repository_url' => $repository->url,
                    'error' => "Не удалось подключиться к репозиторию {$repository->url}: " . $e->getMessage(),
                    'operating_systems' => []
                ];
            }
        }

        // Возвращаем представление с обоими списками
        return view('template_os.index', compact('templateOsList', 'osListFromRepositories'));
    }


    // Загрузка ОС
    public function download(Request $request, $id)
    {
        $templateOs = TemplateOS::find($id);

        // Запуск job для скачивания и установки ОС
        DownloadAndInstallOs::dispatch($templateOs);

        return redirect()->route('template_os.index')->with('success', 'ОС загружается и устанавливается.');
    }

    // Переустановка ОС
    public function reinstall(Request $request, $id)
    {
        $templateOs = TemplateOS::find($id);

        // Удаление и повторная установка ОС через job
        DeleteOsJob::dispatch($templateOs);
        DownloadAndInstallOs::dispatch($templateOs);

        return redirect()->route('template_os.index')->with('success', 'ОС переустанавливается.');
    }

    // Удаление ОС
    public function delete(Request $request, $id)
    {
        $templateOs = TemplateOS::find($id);

        // Удаление ОС через job
        DeleteOsJob::dispatch($templateOs);

        return redirect()->route('template_os.index')->with('success', 'ОС удаляется.');
    }

    // Обновление списка ОС
    public function refresh()
    {
        $repositories = Repository::all();

        foreach ($repositories as $repository) {
            // Получаем данные из info.json
            $response = Http::get($repository->url . '/info.json');
            $osList = $response->json();

            // Проходим по каждой операционной системе в info.json
            foreach ($osList['operating_systems'] as $osData) {
                $name = $osData['name'];
                $tags = implode(' ', $osData['tags']); // Преобразуем теги в строку для записи

                // Проверяем, существует ли ОС локально
                $localPath = storage_path('os/' . $name);
                $localSize = Storage::exists($localPath) ? $this->getDirectorySize($localPath) : null;

                // Обновляем информацию о ОС в базе данных
                TemplateOS::updateOrCreate(
                    ['name' => $name],
                    [
                        'cpu_min' => $osData['cpu_min'] ?? null, // Минимальные требования CPU (если есть)
                        'ram_min' => $osData['ram_min'] ?? null, // Минимальные требования RAM
                        'disk_min' => $osData['disk_min'] ?? null, // Минимальный объем диска
                        'status' => $localSize ? 'installed' : 'available',
                        'local_size' => $localSize,
                        'archive_size' => $osData['archive_size'] ?? null, // Размер архива из info.json
                    ]
                );
            }
        }

        return redirect()->route('template_os.index')->with('success', 'Список ОС обновлён.');
    }


    // Функция для расчёта размера локальной папки
    private function getDirectorySize($path)
    {
        $size = 0;
        foreach (Storage::allFiles($path) as $file) {
            $size += Storage::size($file);
        }
        return $this->formatBytes($size);
    }

    // Форматирование размера в человеко-читаемый вид
    private function formatBytes($size, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];

        $base = log($size, 1024);
        $suffix = $units[floor($base)];

        return round(pow(1024, $base - floor($base)), $precision) . ' ' . $suffix;
    }

    public function downloadFromRepo(Request $request)
    {
        $osFiles = $request->input('os_files');
        $osNames = $request->input('os_names'); // предполагается, что в запросе будут и имена ОС
        $cpuMinValues = $request->input('cpu_min');
        $ramMinValues = $request->input('ram_min');
        $diskMinValues = $request->input('disk_min');

        // Проходим по каждому выбранному файлу ОС и добавляем его в очередь загрузки
        foreach ($osFiles as $index => $file) {
            $osName = $osNames[$index];
            $cpuMin = $cpuMinValues[$index];
            $ramMin = $ramMinValues[$index];
            $diskMin = $diskMinValues[$index];

            // Создаем запись об ОС
            $os = TemplateOS::create([
                'name' => $osName,
                'status' => 'downloading',
                'archive_name' => $file,
                'cpu_min' => $cpuMin,
                'ram_min' => $ramMin,
                'disk_min' => $diskMin,
            ]);

            // Запускаем задачу для каждой ОС
            DownloadAndInstallOs::dispatch($os);
        }

        return redirect()->route('template_os.index')->with('success', 'Выбранные ОС загружаются и устанавливаются.');
    }

    public function downloadMultipleFromRepo(Request $request)
    {
        $osFiles = $request->input('os_files'); // Получаем массив файлов ОС
        $osNames = $request->input('os_names'); // Получаем имена ОС
        $cpuMinValues = $request->input('cpu_min');
        $ramMinValues = $request->input('ram_min');
        $diskMinValues = $request->input('disk_min');

        // Проходим по каждому выбранному файлу ОС и добавляем его в очередь загрузки
        foreach ($osFiles as $index => $file) {
            $osName = $osNames[$index];
            $cpuMin = $cpuMinValues[$index];
            $ramMin = $ramMinValues[$index];
            $diskMin = $diskMinValues[$index];

            // Создаем запись об ОС
            $os = TemplateOS::create([
                'name' => $osName,
                'status' => 'downloading',
                'archive_name' => $file,
                'cpu_min' => $cpuMin,
                'ram_min' => $ramMin,
                'disk_min' => $diskMin,
            ]);

            // Запускаем задачу для каждой ОС
            DownloadAndInstallOs::dispatch($os);
        }

        return redirect()->route('template_os.index')->with('success', 'Выбранные ОС загружаются и устанавливаются.');
    }



}
