<?php

namespace App\Http\Controllers;

use App\Models\Subnet;
use App\Models\IpAddress;
use Illuminate\Http\Request;

class SubnetController extends Controller
{
    public function index()
    {
        $subnets = Subnet::all();
        return view('subnets.index', compact('subnets'));
    }

    public function create()
    {
        return view('subnets.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'subnet' => 'required|string',
            'type' => 'required|in:ipv4,ipv6',
            'is_ip_list' => 'boolean',
        ]);

        // Если это список IP-адресов
        if ($request->has('is_ip_list')) {
            // Разделяем строку по запятым для списка IP-адресов
            $ips = array_map('trim', explode(',', $request->input('subnet')));

            foreach ($ips as $ip) {
                // Проверяем корректность каждого IP-адреса
                if (!filter_var($ip, FILTER_VALIDATE_IP)) {
                    throw new \Exception('Invalid IP address: ' . $ip);
                }

                // Создаём запись для каждого IP-адреса в таблице ip_addresses
                IpAddress::create([
                    'subnet_id' => null,  // Если нет привязки к подсети
                    'ip_address' => $ip,
                    'type' => $request->input('type'),
                    'status' => 'available',
                ]);
            }

            return redirect()->route('subnets.index')->with('success', 'IP list created successfully');
        }

        // Если это целая подсеть
        $totalIps = Subnet::calculateTotalIps($request->input('subnet'));

        // Создание подсети
        $subnet = Subnet::create([
            'subnet' => $request->input('subnet'),
            'type' => $request->input('type'),
            'is_ip_list' => false,
            'total_ips' => $totalIps,
        ]);

        return redirect()->route('subnets.index')->with('success', 'Subnet created successfully');
    }


    public function edit(Subnet $subnet)
    {
        $ipAddresses = IpAddress::where('subnet_id', $subnet->id)->get();

        // Если это список IP-адресов
        if ($subnet->is_ip_list) {
            return view('subnets.ip_list', compact('subnet', 'ipAddresses'));
        }

        // Открываем страницу редактирования обычной подсети
        return view('subnets.edit', compact('subnet', 'ipAddresses'));
    }

    public function update(Request $request, Subnet $subnet)
    {
        $request->validate([
            'subnet' => 'required|string',
            'type' => 'required|in:ipv4,ipv6',
            'total_ips' => 'required|integer',
            'reserved_ips' => 'nullable|string',
        ]);

        // Обновление информации о подсети
        $subnet->update([
            'subnet' => $request->input('subnet'),
            'type' => $request->input('type'),
            'total_ips' => $request->input('total_ips'),
        ]);

        // Обработка зарезервированных IP
        if ($request->filled('reserved_ips')) {
            $reservedIps = array_map('trim', explode(',', $request->input('reserved_ips')));

            // Сначала снять резервирование с всех IP, связанных с этой подсетью
            IpAddress::where('subnet_id', $subnet->id)
                ->where('status', 'reserved')
                ->update(['status' => 'available']);

            // Затем резервировать новые IP
            foreach ($reservedIps as $ip) {
                IpAddress::where('subnet_id', $subnet->id)
                    ->where('ip_address', $ip)
                    ->update(['status' => 'reserved']);
            }
        }

        return redirect()->route('subnets.index')->with('success', 'Subnet updated successfully');
    }

    public function destroy(Subnet $subnet)
    {
        $subnet->delete();
        return redirect()->route('subnets.index')->with('success', 'Subnet deleted successfully');
    }

    public function reserveIp($id)
    {
        $ip = IpAddress::findOrFail($id);

        if ($ip->status == 'available') {
            $ip->update(['status' => 'reserved']);
            return redirect()->back()->with('success', 'IP address reserved successfully');
        }

        return redirect()->back()->with('error', 'IP address cannot be reserved');
    }

    public function releaseIp($id)
    {
        $ip = IpAddress::findOrFail($id);

        if ($ip->status == 'reserved') {
            $ip->update(['status' => 'available']);
            return redirect()->back()->with('success', 'IP address released successfully');
        }

        return redirect()->back()->with('error', 'IP address cannot be released');
    }

    public function saveComment(Request $request, $id)
    {
        $ip = IpAddress::findOrFail($id);
        $ip->update(['comment' => $request->input('comment')]);
        return redirect()->back()->with('success', 'Comment saved');
    }


    public function showIps(Subnet $subnet)
    {
        // Получаем все IP-адреса, относящиеся к данной подсети
        $ipAddresses = $subnet->ipAddresses;

        // Передаём подсеть и IP-адреса в представление
        return view('subnets.ip_list', compact('subnet', 'ipAddresses'));
    }



}
