<?php

namespace App\Http\Controllers;

use App\Models\HostNode;
use App\Models\Pool;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Jobs\InstallSoftwareOnNode;

class HostNodeController extends Controller
{

    /**
     * Конструктор контроллера.
     *
     * Устанавливает middleware для проверки прав доступа к методам контроллера.
     */
//    public function __construct()
//    {
//        $this->middleware('can:view_host_nodes')->only(['index', 'show']);
//        $this->middleware('can:create_host_nodes')->only(['create', 'store']);
//        $this->middleware('can:edit_host_nodes')->only(['edit', 'update']);
//        $this->middleware('can:delete_host_nodes')->only(['destroy']);
//    }

    public function index(Request $request)
    {
        $query = HostNode::query();

        // Поиск по имени и IP-адресу
        if ($request->has('search')) {
            $search = $request->input('search');
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhere('ip_address', 'LIKE', "%{$search}%");
        }

        $hostNodes = $query->get();

        return view('host_nodes.index', compact('hostNodes'));
    }

    public function create()
    {
        $pool = Pool::first(); // Получаем первый пул, который используется всеми узлами

        return view('host_nodes.create', compact('pool'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'ip_address' => 'required|ip|unique:host_nodes',
            'ssh_username' => 'required|string|max:255',
            'ssh_port' => 'required|integer|min:1|max:65535',
            'agent_port' => 'required|integer|min:1|max:65535',
            'disk_device' => 'required_if:pool_type,lvm|string'
            // дополнительные правила валидации при необходимости
        ]);

        // Создание ноды с заданием значений по умолчанию
        $hostNode = HostNode::create([
            'name' => $request->name,
            'ip_address' => $request->ip_address,
            'ssh_username' => $request->ssh_username,
            'ssh_password' => $request->ssh_password,
            'ssh_port' => $request->ssh_port ?? 22,
            'agent_port' => $request->agent_port ?? 1515,
            'auth_key' => Str::random(32),
            'description' => $request->description,
            'is_available' => 0,
            'is_locked' => 1,
            'is_technical' => 0,
            'status' => 'installing'
        ]);

        // Отправка задания в очередь для установки ПО
        InstallSoftwareOnNode::dispatch($hostNode);

        return redirect()->route('host_nodes.index')->with('success', 'Host node created successfully. Installation is in progress.');
    }

    public function show(HostNode $hostNode)
    {
        return view('host_nodes.show', compact('hostNode'));
    }

    public function edit(HostNode $hostNode)
    {
        $pool = Pool::first();
        return view('host_nodes.edit', compact('hostNode', 'pool'));
    }

    public function update(Request $request, HostNode $hostNode)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'ip_address' => 'required|ip|unique:host_nodes,ip_address,' . $hostNode->id,
            'ssh_username' => 'required|string|max:255',
            // добавьте дополнительные правила валидации, если необходимо
        ]);

        $hostNode->update($request->all());

        return redirect()->route('host_nodes.index')->with('success', 'Host node updated successfully');
    }

    public function destroy(HostNode $hostNode)
    {
        $hostNode->delete();

        return redirect()->route('host_nodes.index')->with('success', 'Host node deleted successfully');
    }
}

